<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Rules\PasswordHistory;
use Illuminate\Support\Facades\Password;

class ResetPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset requests
    | and uses a simple trait to include this behavior. You're free to
    | explore this trait and override any methods you wish to tweak.
    |
    */

    use ResetsPasswords;

    protected $passwordLength = '9';

    /**
     * Where to redirect users after resetting their password.
     *
     * @var string
     */
    //protected $redirectTo = '/home';
    protected function redirectTo()
    {

        Auth::logout();
        return '/reset-password-successfully';
        //return '/home';
    }

    protected function guard()
    {
        return Auth::guard('web');
    }

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');

        $this->broker()->validator(function (array $credentials) {

            $passwordHistories = DB::table('password_histories')->where('email', $credentials['email'])->take(env('PASSWORD_HISTORY_NUM'))->get();
            foreach ($passwordHistories as $passwordHistory) {

                if (Hash::check($credentials['password'], $passwordHistory->password)) {
                    // The passwords matches
                    return redirect()->back()->with("error", "Your new password can not be same as any of your recent passwords. Please choose a new password.");
                    //echo "matched";
                }
            }

            [$password, $confirm] = [
                $credentials['password'],
                $credentials['password_confirmation'],
            ];

            return $password === $confirm && mb_strlen($password) >= $this->passwordLength;
        });
    }

    protected function rules()
    {
        return [
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:' . $this->passwordLength . '|regex:/(?=.{9,})((?=.*\d)(?=.*[a-z])|(?=.*\d)(?=.*[a-zA-Z])(?=.*[\W_])|(?=.*[a-z])(?=.*[A-Z])(?=.*[\W_])).*|' . new PasswordHistory,
            ////'password' => 'required|min:' . $this->passwordLength . '|regex:/[a-z]/|regex:/[A-Z]/|regex:/[0-9]/|regex:/[@$!%*#?&]/|' . new PasswordHistory($this->email),
        ];
    }

    public function reset(Request $request)
    {

        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            //'password' => 'required|min:' . $this->passwordLength . '|regex:/^((?=.*?[A-Z])(?=.*?[a-z])(?=.*?[0-9])|(?=.*?[A-Z])(?=.*?[a-z])(?=.*?[^a-zA-Z0-9])|(?=.*?[A-Z])(?=.*?[0-9])(?=.*?[^a-zA-Z0-9])|(?=.*?[a-z])(?=.*?[0-9])(?=.*?[^a-zA-Z0-9])).{9,}$/|password_histories',
            //'password' => ['required', 'min:' . $this->passwordLength, 'regex:/(?=.{9,})((?=.*\d)(?=.*[a-z])|(?=.*\d)(?=.*[a-zA-Z])(?=.*[\W_])|(?=.*[a-z])(?=.*[A-Z])(?=.*[\W_])).*/', 'password_histories'],
            'password' => ['required', 'min:' . $this->passwordLength, 'regex:/((?=.*\d)(?=.*[a-z])(?=.*[A-Z])|(?=.*\d)(?=.*[a-zA-Z])(?=.*[\W_])|(?=.*[a-z])(?=.*[A-Z])(?=.*[\W_])).*/', 'password_histories'],
        ], [
            'password_histories' => 'This password has been used in your last 10 passwords. Please enter another password', // <---- pass a message for your custom validator
        ]);

        // Here we will attempt to reset the user's password. If it is successful we
        // will update the password on an actual user model and persist it to the
        // database. Otherwise we will parse the error and return the response.
        $this->broker()->validator(function ($credentials) {
            return mb_strlen($credentials['password']) >= 9;
        });
        $response = $this->broker()->reset(
            $this->credentials($request),
            function ($user, $password) {
                $this->resetPassword($user, $password);
            }
        );

        // If the password was successfully reset, we will redirect the user back to
        // the application's home authenticated view. If there is an error we can
        // redirect them back to where they came from with their error message.
        return $response == Password::PASSWORD_RESET
            ? $this->sendResetResponse($request, $response)
            : $this->sendResetFailedResponse($request, $response);
    }

    /*
    public function showResetForm(Request $request, $token = null)
    {
        return view('auth.passwords.reset')->with(
            ['token' => $token, 'email' => decrypt($request->email)]
        );
    }
    */
}
